import type { Metadata } from "next";
import { NextIntlClientProvider } from "next-intl";
import { getMessages, setRequestLocale } from "next-intl/server";
import { notFound } from "next/navigation";
import { routing } from "@/core/lib/i18n/routing";
import { getSettings } from "@/core/lib/api";
import { buildMetadataFromSeo } from "@/core/lib/seo";

export function generateStaticParams() {
  return routing.locales.map((locale) => ({ locale }));
}

type Props = {
  children: React.ReactNode;
  params: Promise<{ locale: string }>;
};

export async function generateMetadata({
  params,
}: {
  params: Promise<{ locale: string }>;
}): Promise<Metadata> {
  const { locale } = await params;
  const settings = await getSettings();
  const lang = (locale === "ar" ? "ar" : "en") as "en" | "ar";

  return buildMetadataFromSeo({
    seo: (settings as any)?.seo ?? undefined,
    openGraph: (settings as any)?.openGraph ?? undefined,
    locale: lang,
    override: {
      alternates: {
        canonical: `${process.env.NEXT_PUBLIC_SITE_URL || "https://www.yslootahtech.com"}/${locale}`,
      },
    },
  });
}

export default async function LocaleLayout({ children, params }: Props) {
  const { locale } = await params;

  if (!routing.locales.includes(locale as any)) {
    notFound();
  }

  setRequestLocale(locale);
  const messages = await getMessages();

  return (
    <NextIntlClientProvider messages={messages}>
      <div dir={locale === 'ar' ? 'rtl' : 'ltr'} lang={locale}>
        {children}
      </div>
    </NextIntlClientProvider>
  );
}
